<?php
/**
 * Genesis Framework.
 *
 * Provides update notices to make users aware that Genesis and Genesis child
 * themes do not use the WordPress.org update process.
 *
 * WARNING: This file is part of the core Genesis Framework. DO NOT edit this file under any circumstances.
 * Please do all modifications in the form of a child theme.
 *
 * @package Genesis\Admin
 * @author  StudioPress
 * @license GPL-2.0-or-later
 * @link    https://my.studiopress.com/themes/genesis/
 * @link    https://make.wordpress.org/core/2020/07/15/controlling-plugin-and-theme-auto-updates-ui-in-wordpress-5-5/
 * @link    https://core.trac.wordpress.org/ticket/50663
 */

namespace StudioPress\Genesis\Admin\AutoUpdateNotices;

add_filter( 'theme_auto_update_setting_template', __NAMESPACE__ . '\\setting_template' );
/**
 * Alters the update notice shown with theme information.
 *
 * @since 3.3.3
 *
 * @param string $original_template Original update notice template.
 * @return string The new template.
 */
function setting_template( $original_template ) {
	$genesis_update_html = sprintf(
		/* translators: %s: URL for the Genesis Updates section in the Customizer Theme Settings panel. */
		__( 'Enable update checks in <a href="%s">Genesis update settings</a>.', 'genesis' ),
		admin_url( 'customize.php?autofocus[section]=genesis_updates' )
	);

	$child_theme_update_text = __( 'Auto-updates are not available for this theme.', 'genesis' );

	return "<# if ( 'genesis' === data.id ) { #>
	<p>$genesis_update_html</p>
	<# } else if ( 'Genesis' === data.parent ) { #>
	<p>$child_theme_update_text</p>
	<# } else { #>
	$original_template
	<# } #>";
}

add_filter( 'theme_auto_update_debug_string', __NAMESPACE__ . '\\debug_string', 10, 2 );
/**
 * Alters auto-update notice debug text, such as on the Site Health page.
 *
 * @since 3.3.3
 *
 * @param string $notice Current update notice.
 * @param object $theme Theme data.
 * @return string|void New update notice.
 */
function debug_string( $notice, $theme ) {
	if ( 'Genesis' === $theme->name ) {
		return __( 'Enable update checks at Appearance → Customize → Theme Settings → Updates.', 'genesis' );
	}
	if ( 'genesis' === $theme->template ) {
		return __( 'Auto-updates are not available for this theme.', 'genesis' );
	}
	return $notice;
}

add_filter( 'theme_auto_update_setting_html', __NAMESPACE__ . '\\setting_html', 10, 3 );
/**
 * Alters auto-update notice text in the theme updates listing,
 * visible on WordPress multisite.
 *
 * @since 3.3.3
 *
 * @param string $html Current update HTML.
 * @param string $stylesheet Stylesheet name.
 * @param object $theme Theme data.
 * @return string|void New update HTML.
 */
function setting_html( $html, $stylesheet, $theme ) {
	return \StudioPress\Genesis\Admin\AutoUpdateNotices\debug_string( $html, $theme );
}
